<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . "/includes/auth.php";
require_super(); // التحقق من دخول السوبر أدمن

require_once __DIR__ . "/includes/functions.php";

// جلب اللغات المتاحة (الثابتة) من جدول languages
$all_langs = fetch_all("SELECT * FROM languages ORDER BY id ASC");

// معالجة الإضافة
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $name   = clean($_POST['name'] ?? '');
    $type   = clean($_POST['subscription_type'] ?? 'monthly');
    $days   = (int)($_POST['days'] ?? 0);
    $active = isset($_POST['active']) ? 1 : 1; // افتراضياً فعال

    // ✅ الرسوم
    $fees = (float)($_POST['fees'] ?? 0);

    // ✅ تفعيل الميزات (0/1)
    $reservations_enabled = isset($_POST['reservations_enabled']) ? 1 : 0;
    $orders_enabled       = isset($_POST['orders_enabled']) ? 1 : 0;
    $reports_enabled      = isset($_POST['reports_enabled']) ? 1 : 0;
    $discounts_enabled    = isset($_POST['discounts_enabled']) ? 1 : 0;

    // بيانات المدير
    $manager_name  = clean($_POST['manager_name'] ?? '');
    $manager_email = clean($_POST['manager_email'] ?? '');
    $manager_phone = clean($_POST['manager_phone'] ?? '');
    $manager_pass  = $_POST['manager_password'] ?? '';

    // لغات المطعم (قيمها = language_id من جدول languages)
    $selected_lang_ids = $_POST['languages'] ?? [];

    if ($name === '' || $manager_email === '' || $manager_pass === '') {
        die("الاسم والإيميل وكلمة المرور حقول إلزامية.");
    }

    // توليد اسم مجلد (folder_name) بناءً على اسم المطعم
    $folder_base = $name;
    $folder_base = strtolower($folder_base);
    $folder_base = preg_replace('/[^a-z0-9]+/', '-', $folder_base);
    $folder_base = trim($folder_base, '-');

    if ($folder_base === '') {
        $folder_base = 'restaurant-' . time();
    }

    $folder_name = $folder_base;

    // حساب تاريخ البداية والنهاية للاشتراك
    $start = new DateTime(); // اليوم
    $end   = clone $start;

    switch ($type) {
        case 'yearly':
            $end->modify('+365 days');
            break;

        case 'fixed': // ✅ مطابق للـ enum بقاعدتك
            if ($days <= 0) $days = 30;
            $end->modify('+' . $days . ' days');
            break;

        case 'monthly':
        default:
            $end->modify('+30 days');
            $type = 'monthly';
            break;
    }

    $subscription_start = $start->format('Y-m-d');
    $subscription_end   = $end->format('Y-m-d');

    try {
        db()->beginTransaction();

        // إدخال المطعم في جدول restaurants
        $stmt = db()->prepare("
            INSERT INTO restaurants 
            (
                name,
                folder_name,
                subscription_type,
                subscription_start,
                subscription_end,
                active,
                reservations_enabled,
                orders_enabled,
                reports_enabled,
                discounts_enabled,
                fees
            )
            VALUES 
            (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");

        $stmt->execute([
            $name,
            $folder_name,
            $type,
            $subscription_start,
            $subscription_end,
            $active,
            $reservations_enabled,
            $orders_enabled,
            $reports_enabled,
            $discounts_enabled,
            $fees
        ]);

        $restaurant_id = (int)db()->lastInsertId();

        // ربط لغات المطعم في جدول restaurant_languages
        if (!empty($selected_lang_ids) && $restaurant_id > 0) {
            $insLang = db()->prepare("
                INSERT INTO restaurant_languages (restaurant_id, language_id, active)
                VALUES (?, ?, 1)
            ");

            foreach ($selected_lang_ids as $lang_id) {
                $lang_id = (int)$lang_id;
                if ($lang_id > 0) {
                    $insLang->execute([$restaurant_id, $lang_id]);
                }
            }
        }

        // إنشاء مدير المطعم في جدول managers
        if ($restaurant_id > 0) {
            $password_hash = password_hash($manager_pass, PASSWORD_BCRYPT);

            $stmtM = db()->prepare("
                INSERT INTO managers (restaurant_id, name, email, phone, password, role, active)
                VALUES (?, ?, ?, ?, ?, 'manager', 1)
            ");
            $stmtM->execute([
                $restaurant_id,
                $manager_name,
                $manager_email,
                $manager_phone,
                $password_hash
            ]);
        }

        // إنشاء مجلد المطعم ونسخ مجلد Demo الأمامي إليه
        create_restaurant_folder($folder_name);

        db()->commit();

        header("Location: restaurants.php");
        exit;

    } catch (Exception $e) {
        db()->rollBack();
        die("خطأ في إنشاء المطعم: " . $e->getMessage());
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
<meta charset="UTF-8">
<title>إضافة مطعم جديد</title>
<meta name="viewport" content="width=device-width, initial-scale=1">

<style>
body{
    margin:0;
    padding:0;
    background:#0f0f11;
    color:#fff;
    font-family:"Tajawal",sans-serif;
    display:flex;
}
.content{
    margin-right:300px;
    padding:35px;
    width:100%;
}
h1{
    color:#9c6bff;
    margin-bottom:20px;
}
.form-box{
    background:#1b1b22;
    padding:25px;
    border-radius:12px;
    border:1px solid #2a2933;
    max-width:800px;
}
.form-row{
    display:flex;
    gap:15px;
}
.form-group{
    margin-bottom:15px;
    flex:1;
}
label{
    display:block;
    margin-bottom:6px;
}
input, select{
    width:100%;
    padding:10px;
    border-radius:8px;
    border:1px solid #333;
    background:#111;
    color:#fff;
}
.checkbox-list{
    display:flex;
    flex-wrap:wrap;
    gap:10px;
}
.checkbox-list label{
    display:flex;
    align-items:center;
    gap:5px;
    background:#111;
    padding:8px 10px;
    border-radius:8px;
}
button{
    padding:10px 20px;
    border:none;
    border-radius:8px;
    background:#9c6bff;
    color:#000;
    font-weight:bold;
    cursor:pointer;
}
button:hover{
    opacity:0.9;
}
</style>
</head>
<body>

<?php include "includes/sidebar.php"; ?>

<div class="content">
    <h1>إضافة مطعم جديد</h1>

    <div class="form-box">
        <form method="post">

            <div class="form-group">
                <label>اسم المطعم</label>
                <input type="text" name="name" required>
            </div>

            <h3>اللغات المتاحة للمطعم</h3>
            <div class="form-group">
                <div class="checkbox-list">
                    <?php foreach ($all_langs as $lang): ?>
                        <label>
                            <input type="checkbox" name="languages[]" value="<?= $lang['id'] ?>">
                            <?= $lang['name'] ?> (<?= $lang['code'] ?>)
                        </label>
                    <?php endforeach; ?>
                </div>
            </div>

            <hr style="border-color:#333; margin:20px 0;">

            <h3>تفعيل الميزات</h3>
            <div class="form-group">
                <div class="checkbox-list">
                    <label>
                        <input type="checkbox" name="reservations_enabled" value="1">
                        تفعيل الحجوزات
                    </label>

                    <label>
                        <input type="checkbox" name="orders_enabled" value="1">
                        تفعيل الطلبات
                    </label>

                    <label>
                        <input type="checkbox" name="reports_enabled" value="1">
                        تفعيل التقارير
                    </label>
                    <label>
                        <input type="checkbox" name="discounts_enabled" value="1">
                        تفعيل الخصومات
                    </label>
                </div>
            </div>

            <hr style="border-color:#333; margin:20px 0;">

            <!-- ✅ الرسوم -->
            <h3>الرسوم</h3>
            <div class="form-row">
                <div class="form-group">
                    <label>رسوم الحساب (مثلاً بالدولار أو بالدينار حسب نظامك)</label>
                    <input type="number" name="fees" step="0.01" min="0" value="0">
                </div>
            </div>

            <hr style="border-color:#333; margin:20px 0;">

            <h3>بيانات الاشتراك</h3>
            <div class="form-row">
                <div class="form-group">
                    <label>نوع الاشتراك</label>
                    <select name="subscription_type" id="sub_type" onchange="toggleDays()">
                        <option value="monthly">شهري</option>
                        <option value="yearly">سنوي</option>
                        <option value="fixed">عدد أيام محدد</option>
                    </select>
                </div>
                <div class="form-group" id="days_box" style="display:none;">
                    <label>عدد الأيام</label>
                    <input type="number" name="days" min="1" value="30">
                </div>
            </div>

            <hr style="border-color:#333; margin:20px 0;">

            <h3>مدير المطعم</h3>
            <div class="form-row">
                <div class="form-group">
                    <label>اسم المدير</label>
                    <input type="text" name="manager_name" required>
                </div>
                <div class="form-group">
                    <label>البريد الإلكتروني</label>
                    <input type="email" name="manager_email" required>
                </div>
            </div>

            <div class="form-row">
                <div class="form-group">
                    <label>رقم الهاتف</label>
                    <input type="text" name="manager_phone">
                </div>
                <div class="form-group">
                    <label>كلمة المرور</label>
                    <input type="password" name="manager_password" required>
                </div>
            </div>

            <br>
            <button type="submit">حفظ المطعم</button>
        </form>
    </div>
</div>

<script>
function toggleDays(){
    const type = document.getElementById('sub_type').value;
    const box  = document.getElementById('days_box');
    box.style.display = (type === 'fixed') ? 'block' : 'none';
}
</script>

</body>
</html>
